//******************************************************************************
//  MSP430x26x Demo - SMBus Master Interface to SMBus Slave TMP175
//
//  Description: I2C/SMBus interface to TMP175 temperature sensor in 9-bit mode. 
//  TimerA0 CCR0 interrupt is used as a wake up and to read
//  the TMP175 temperatre register. If the temperature is greater than 28c,
//  P1.0 is set, else reset.  CPU is normally off and used only during TA_ISR.
//  P2.0 is used for responding to the ALERT response from the Slave TMP175.
//  A Slave Alert Response is sent by the master and TMP175 puts its address
//  on the bus.
//   ACLK = n/a, MCLK = SMCLK = BRCLK = DCO ~ 1.1MHz
//   //* USCI Required *//
//
//                           /|\
//         /|\           /|\  |  /|\
//          |   TMP175   10k 10K 10k     MSP430F2619           
//          |   -------   |   |   |   ----------------- 
//          +--|Vcc SDA|<-|---+---|->|P3.1         P1.0|--->LED
//             |       |  |       |  |                 |
//          +-|A2,A1,A0|  |       |  |                 |
//          |  |  ALERT|--|-------+->|P2.0             |
//          +--|Vss SCL|<-+----------|P3.2             |
//         \|/  -------              |                 |     
//
//
//  H. Grewal
//  Texas Instruments Inc.
//  June 2009
//  Built with IAR Embedded Workbench Version: 4.11B
//******************************************************************************

#include  <msp430x26x.h>
typedef enum {
             SMBS_MISC               = (unsigned char) 0x48,
             SMBS_DEVICE_DFLT        = (unsigned char) 0x61,
             SMBS_ALERT_RESPONSE     = (unsigned char) 0x19,
             SMBS_HOST               = (unsigned char) 0x08,
             SMBS_10BIT_ADD          = (unsigned char) 0x78,
             SMBS_DFLT_ADD           = (unsigned char) 0x37,
             SMBS_ACCESS_HOST        = (unsigned char) 0x28,             
             SMBS_CBUS               = (unsigned char) 0x01,
             SMBS_GCAL               = (unsigned char) 0x00
             }SMBS_Address_t;               
/*---------------------------------------------------------------------------*/
unsigned char *PTxData;                     // Pointer to TX data
unsigned char TXByteCtr;
const unsigned char TxData[] =              // Table of data to transmit
{
  0x03,                                     // Load  Pointer register--T(high)
  0x10,                                     // Load  T(high) register, @16 C
  0x01,                                     // Load  Pointer register--Config reg
  0x02,                                     // Load  Configuration register--TM mode
  0x00                                      // Load  Pointer reg--Temp reg READ
};

void SMBS_Init (SMBS_Address_t Add_Param,unsigned char Slave_Add);
void setupTMP175_1 (void);
void setupTMP175_2 (void);

void main(void)
{
  WDTCTL = WDTPW + WDTHOLD;                 // Stop watchdog
  P1DIR |= 0xFF;                            // Set P1.0 to output direction
                                            // unused ports to output
  P3DIR =  0xFF;                                            
  P3SEL |= 0x06;                            // Assign I2C pins to module
  P4DIR =  0xFF;
  P5DIR =  0xFF;
  P6DIR =  0xFF;
  SMBS_Init (SMBS_MISC,0x48);               // Slave address configured as 48h
  
  PTxData = (unsigned char *)TxData;        // TX array start address
  setupTMP175_1();                          // Indicate pointer reg, load T(high)
  setupTMP175_1();                          // Indicate pointer reg, load Config
  P2DIR = 0xFE;                             // set P2.0 to input  
  P2IES = 0x01;                             // Interrupt Falling edge select 
  P2IFG = 0;                                // Clear pending P2 interrupts
  P2IE = 0x01;                              // Enable P2.0 interrupt
  setupTMP175_2();                          // Indicate pointer reg as Temp reg
  
  CCTL0 = CCIE;                             // CCR0 interrupt enabled
  TACTL = TASSEL_2 + MC_2;                  // SMCLK, continuous mode

  __bis_SR_register(LPM0_bits + GIE)        // Enter LPM0 w/ interrupt
}
void SMBS_Init (SMBS_Address_t Add_Param,unsigned char Slave_Add)
{
  UCB0CTL1 |= UCSWRST;                      // Enable SW reset
  UCB0CTL0 = UCMST + UCMODE_3 + UCSYNC;     // I2C Master, synchronous mode
  UCB0CTL1 = UCSSEL_2 + UCSWRST;            // Use SMCLK, keep SW reset
  UCB0BR0 = 12;                             // fSCL = SMCLK/12 = ~100kHz
  UCB0BR1 = 0;
  if (Add_Param == SMBS_MISC)
  {
      UCB0I2CSA = Slave_Add;                // Slave Address is 048h
  } 
  else
      UCB0I2CSA = (unsigned char) Add_Param ;  
 UCB0CTL1 &= ~UCSWRST;                      // Clear SW reset, resume operation
 IE2 |= UCB0TXIE;                           // Enable RX interrupt
  
}

void setupTMP175_1(void)
{
  TXByteCtr = 2;                            // Load TX byte counter
  while (UCB0CTL1 & UCTXSTP);               // Ensure stop condition got sent

  UCB0CTL1 |= UCTR + UCTXSTT;               // I2C TX, start condition
  __bis_SR_register(LPM0_bits + GIE);       // Enter LPM0 w/ interrupts
                                            // Remain in LPM0 until all data
                                            // is TX'd  
}

void setupTMP175_2(void)
{
  TXByteCtr = 1;                            // Load TX byte counter
  while (UCB0CTL1 & UCTXSTP);               // Ensure stop condition got sent

  UCB0CTL1 |= UCTR + UCTXSTT;               // I2C TX, start condition
  __bis_SR_register(LPM0_bits + GIE);       // Enter LPM0 w/ interrupts
                                            // Remain in LPM0 until all data
                                            // is TX'd  
  UCB0CTL1 &= ~UCTR;                        // Master receiver to read temperature
  IE2 &= ~UCB0TXIE;                         // Enable RX interrupt
}


//------------------------------------------------------------------------------
// The USCIAB0TX_ISR is structured such that it can be used to transmit any
// number of bytes by pre-loading TXByteCtr with the byte count. Also, TXData
// points to the next byte to transmit.
//------------------------------------------------------------------------------
#pragma vector = USCIAB0TX_VECTOR
__interrupt void USCIAB0TX_ISR(void)
{
  if (TXByteCtr)                            // Check TX byte counter
  {
    UCB0TXBUF = *PTxData++;                 // Load TX buffer
    TXByteCtr--;                            // Decrement TX byte counter
  }
  else
  {
    UCB0CTL1 |= UCTXSTP;                    // I2C stop condition
    IFG2 &= ~UCB0TXIFG;                     // Clear USCI_B0 TX int flag
    __bic_SR_register_on_exit(CPUOFF);      // Exit LPM0
  }
}


// Timer A0 interrupt service routine
#pragma vector=TIMERA0_VECTOR
__interrupt void Timer_A (void)
{
  if (UCB0RXBUF > 0x20) P1OUT |= 0x01;      // LED on if >32C
  else P1OUT &= ~0x01;                      // LED off
  UCB0CTL0 |= UCMST;                        // Master mode
  while (UCB0CTL1 & UCTXSTP);               // Ensure stop condition got sent
  UCB0CTL1 |= UCTXSTT;                      // I2C start condition
  while (UCB0CTL1 & UCTXSTT);               // Start condition sent?
  UCB0CTL1 |= UCTXSTP;                      // I2C stop condition
}
#pragma vector=PORT2_VECTOR
__interrupt void p2_isr(void)
{
  P2IFG = 0;
  UCB0I2CSA = 0x19;                         // SMBus Alert response
  UCB0CTL0 |= UCMST;                        // Master mode
  while (UCB0CTL1 & UCTXSTP);               // Ensure stop condition got sent
  UCB0CTL1 |= UCTXSTT;                      // I2C start condition
  while (UCB0CTL1 & UCTXSTT);               // Start condition sent?
  UCB0CTL1 |= UCTXSTP;                      // I2C stop condition
  UCB0I2CSA = 0x48;                         // Slave Address 
}
